//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// $Id: G4UIbatch.cc,v 1.17 2008/11/21 10:54:16 kmura Exp $
// GEANT4 tag $Name: geant4-09-02 $
//
// ====================================================================
//   G4UIbatch.cc
//
// ====================================================================
#include "G4UIbatch.hh"
#include "G4UImanager.hh"
#include <vector>

////////////////////////////////////////////////////////////////////////
static void Tokenize(const G4String& str, std::vector<G4String>& tokens)
////////////////////////////////////////////////////////////////////////
{
  const char* delimiter= " ";

  str_size pos0= str.find_first_not_of(delimiter);
  str_size pos = str.find_first_of(delimiter, pos0);
  
  while (pos != G4String::npos || pos0 != G4String::npos) {
    if (str[pos0] == '\"') {
      pos = str.find_first_of("\"", pos0+1);
      if(pos != G4String::npos) pos++;
    }
    if (str[pos0] == '\'') {
      pos = str.find_first_of("\'", pos0+1);
      if(pos != G4String::npos) pos++;
    }

    tokens.push_back(str.substr(pos0, pos-pos0));
    pos0 = str.find_first_not_of(delimiter, pos);
    pos = str.find_first_of(delimiter, pos0);
  }
}

// ====================================================================
//
// class description
//
// ====================================================================

////////////////////////////////////////////////////////////////////
G4UIbatch::G4UIbatch(const char* fileName, G4UIsession* prevSession) 
  : previousSession(prevSession), isOpened(false)
////////////////////////////////////////////////////////////////////
{
  macroStream.open(fileName, std::ios::in);
  if(macroStream.fail()) {
    G4cerr << "***** Can not open a macro file <" 
           << fileName << ">" 
           << G4endl;
  } else {
    isOpened= true;
  }

  G4UImanager::GetUIpointer()-> SetSession(this);
}


///////////////////////
G4UIbatch::~G4UIbatch() 
///////////////////////
{
  if(isOpened) macroStream.close();
}


/////////////////////////////////
G4String G4UIbatch::ReadCommand()
/////////////////////////////////
{
  enum { BUFSIZE= 4096 };
  static char linebuf[BUFSIZE];

  G4String cmdtotal= "";
  G4bool qcontinued= false;
  while(macroStream.good()) {
    macroStream.getline(linebuf, BUFSIZE);

    G4String cmdline(linebuf);

    // TAB-> ' ' conversion
    str_size nb=0;
    while ((nb= cmdline.find('\t',nb)) != G4String::npos) {
      cmdline.replace(nb, 1, " ");
    }

    // strip
    cmdline= cmdline.strip(G4String::both);

    // skip null line if single line
    if(!qcontinued && cmdline.size()==0) continue;

    // '#' is treated as echoing something
    if(cmdline[(size_t)0]=='#') return cmdline;

    // tokenize...
    std::vector<G4String> tokens;
    Tokenize(cmdline, tokens);
    qcontinued= false;
    for (G4int i=0; i< G4int(tokens.size()); i++) {
      // string after '#" is ignored
      if(tokens[i][(size_t)0] == '#' ) break;
      // '\' or '_' is treated as continued line.
      if(tokens[i] == '\\' || tokens[i] == '_' ) {
        qcontinued= true;
        // check nothing after line continuation character
        if( i != G4int(tokens.size())-1) {
          G4Exception("unexpected character after "
                      "line continuation character");
        }
        break; // stop parsing
      }
      cmdtotal+= tokens[i];
      cmdtotal+= " ";
    }

    if(qcontinued) continue; // read the next line

    if(cmdtotal.size() != 0) break;
    if(macroStream.eof()) break;
  }

  // strip again
  cmdtotal= cmdtotal.strip(G4String::both);

  // finally,
  if(macroStream.eof() && cmdtotal.size()==0) {
    return "exit";
  }

  return cmdtotal;
}


/////////////////////////////////////////////////////
G4int G4UIbatch::ExecCommand(const G4String& command)
/////////////////////////////////////////////////////
{
  G4UImanager* UI= G4UImanager::GetUIpointer();
  G4int rc= UI-> ApplyCommand(command);

  switch(rc) {
  case fCommandSucceeded:
    break;
  case fCommandNotFound:
    G4cerr << "***** COMMAND NOT FOUND <"
           << command << "> *****" << G4endl;
    break;
  case fIllegalApplicationState:
    G4cerr << "***** Illegal application state <"
           << command << "> *****" << G4endl;
    break;
  default:
    G4int pn= rc%100;
    G4cerr << "***** Illegal parameter (" << pn << ") <"
           << command << "> *****" << G4endl;
  }

  return rc;
}


///////////////////////////////////////
G4UIsession * G4UIbatch::SessionStart() 
///////////////////////////////////////
{
  if(!isOpened) return previousSession;

  while(1) {
    G4String newCommand = ReadCommand();

    if(newCommand == "exit") { 
      break;
    }

    // just echo something
    if( newCommand[(size_t)0] == '#') { 
      if(G4UImanager::GetUIpointer()-> GetVerboseLevel()==2) {
        G4cout << newCommand << G4endl; 
      }
      continue;
    }

    // execute command
    G4int rc= ExecCommand(newCommand);
    if(rc != fCommandSucceeded) {
      G4cerr << G4endl << "***** Batch is interupted!! *****" << G4endl;
      break;
    }
  }

  return previousSession;
}


//////////////////////////////////////////////////
void G4UIbatch::PauseSessionStart(G4String Prompt) 
//////////////////////////////////////////////////
{
  G4cout << "Pause session <" << Prompt << "> start." << G4endl;

  SessionStart();

  G4cout << "Pause session <" << Prompt << "> Terminate." << G4endl;
}

