// This code implementation is the intellectual property of
// the GEANT4 collaboration.
//
// By copying, distributing or modifying the Program (or any work
// based on the Program) you indicate your acceptance of this statement,
// and all its terms.
//
// $Id: G4SteppingVerbose.cc,v 1.5.6.1 1999/12/07 20:53:06 gunter Exp $
// GEANT4 tag $Name: geant4-01-00 $
//
//
//---------------------------------------------------------------
//
// G4SteppingVerbose.cc
//
// Description:
//    Implementation of  the G4SteppingVerbose class
// Contact:
//   Questions and comments to this code should be sent to
//     Katsuya Amako  (e-mail: Katsuya.Amako@kek.jp)
//     Takashi Sasaki (e-mail: Takashi.Sasaki@kek.jp)
//
//---------------------------------------------------------------

#include "G4SteppingVerbose.hh"
#include "G4SteppingManager.hh"

#include "G4VSensitiveDetector.hh"    // Include from 'hits/digi'
#include "G4StepStatus.hh"    // Include from 'tracking'

///#define G4_USE_G4BESTUNIT_FOR_VERBOSE 1

#ifdef G4_USE_G4BESTUNIT_FOR_VERBOSE
#include "G4UnitsTable.hh"
#else
#define G4BestUnit(a,b) a
#endif

//////////////////////////////////////////////////
G4SteppingVerbose::G4SteppingVerbose()
//////////////////////////////////////////////////
{
#ifdef G4_TRACKING_DEBUG
   G4cout << "G4SteppingVerbose has instantiated" << endl;
#endif
}

//////////////////////////////////////////////////
G4SteppingVerbose::~G4SteppingVerbose()
//////////////////////////////////////////////////
{
}

//////////////////////////////////////////////////
void G4SteppingVerbose::NewStep()
//////////////////////////////////////////////////
{
}

//////////////////////////////////////////////////
void G4SteppingVerbose::AtRestDoItInvoked()
//////////////////////////////////////////////////
 {
   G4VProcess* ptProcManager;

   CopyState();

   if(verboseLevel >= 3 ){     
     G4int npt=0;
     G4cout << " **List of AtRestDoIt invoked:" << endl;
     for(size_t np=0; np < MAXofAtRestLoops; np++){
       size_t npGPIL = MAXofAtRestLoops-np-1;
       if( (*fSelectedAtRestDoItVector)(npGPIL) == 2 ){
	 npt++;                
	 ptProcManager = (*fAtRestDoItVector)(np);
	 G4cout << "   # " << npt << " : " 
	   << ptProcManager->GetProcessName() 
	     << " (Forced)" << endl;
       } else if ( (*fSelectedAtRestDoItVector)(npGPIL) == 1 ){
	 npt++;                
	 ptProcManager = (*fAtRestDoItVector)(np);
	 G4cout << "   # " << npt << " : " 
	   << ptProcManager->GetProcessName() << endl;
       }
     }
     
     G4cout << "   Generated secondries # : " << fN2ndariesAtRestDoIt << endl;
     
     if( fN2ndariesAtRestDoIt > 0 ){
       G4cout << "   -- List of secondaries generated : " 
	 << "(x,y,z,kE,t,PID) --" << endl; 
       for( G4int lp1=(*fSecondary).entries()-fN2ndariesAtRestDoIt; 
	   lp1<(*fSecondary).entries(); lp1++){
	 G4cout << "      "
		<< setw( 9)
		<< G4BestUnit((*fSecondary)[lp1]->GetPosition().x(),"Length") << " "
		<< setw( 9)
		<< G4BestUnit((*fSecondary)[lp1]->GetPosition().y(),"Length") << " "
		<< setw( 9)
		<< G4BestUnit((*fSecondary)[lp1]->GetPosition().z(),"Length") << " "
		<< setw( 9)
		<< G4BestUnit((*fSecondary)[lp1]->GetKineticEnergy(),"Energy") << " "
		<< setw( 9)
		<< G4BestUnit((*fSecondary)[lp1]->GetGlobalTime(),"Time") << " "
		<< setw(18)
		<< (*fSecondary)[lp1]->GetDefinition()
	                             ->GetParticleName();
	 G4cout << endl;
       }
     }
   }
   
   if( verboseLevel >= 4 ){ 
     ShowStep();
     G4cout << endl;
   }
}
/////////////////////////////////////////////////////
void G4SteppingVerbose::AlongStepDoItAllDone()
/////////////////////////////////////////////////////
{
   G4VProcess* ptProcManager;

   CopyState();

     if(verboseLevel >= 3){ 
        G4cout << endl;
        G4cout << " >>AlongStepDoIt (after all invocations):" << endl;
        G4cout << "    ++List of invoked processes " << endl;
 
        for(size_t ci=0; ci<MAXofAlongStepLoops; ci++){
            ptProcManager = (*fAlongStepDoItVector)(ci);
            G4cout << "      " << ci+1 << ") ";
            if(ptProcManager != NULL){
               G4cout << ptProcManager->GetProcessName() << endl;
            }
        }         

        ShowStep();
        G4cout << endl;
        G4cout << "    ++List of secondaries generated " 
             << "(x,y,z,kE,t,PID):"
             << "  No. of secodaries = " 
             << (*fSecondary).entries() << endl;

        if((*fSecondary).entries()>0){
           for(G4int lp1=0; lp1<(*fSecondary).entries(); lp1++){
               G4cout << "      "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().x(),"Length") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().y(),"Length") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().z(),"Length") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetKineticEnergy(),"Energy") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetGlobalTime(),"Time")  << " "
                    << setw(18)
                    << (*fSecondary)[lp1]->GetDefinition()
                                         ->GetParticleName();
               G4cout << endl;
	   }
	}
     }
}
////////////////////////////////////////////////////
void G4SteppingVerbose::PostStepDoItAllDone()
////////////////////////////////////////////////////
{
   G4VProcess* ptProcManager;

   CopyState();

   if(fStepStatus != fPostStepDoItProc) return;

   if(verboseLevel >= 3){ 
        G4int npt=0;
        G4cout << endl;
        G4cout << " **PostStepDoIt (after all invocations):" << endl;
        G4cout << "    ++List of invoked processes " << endl;

        for(size_t np=0; np < MAXofPostStepLoops; np++){
	    size_t npGPIL = MAXofPostStepLoops-np-1;
            if((*fSelectedPostStepDoItVector)(npGPIL) == 2){
               npt++;                
               ptProcManager = (*fPostStepDoItVector)(np);
               G4cout << "      " << npt << ") " 
                    << ptProcManager->GetProcessName()  
                    << " (Forced)" << endl;
	     } else if ((*fSelectedPostStepDoItVector)(npGPIL) == 1){
               npt++;                
               ptProcManager = (*fPostStepDoItVector)(np);
               G4cout << "      " << npt << ") " 
                    << ptProcManager->GetProcessName() << endl;
	     }
	  }

        ShowStep();
        G4cout << endl;
        G4cout << "    ++List of secondaries generated " 
             << "(x,y,z,kE,t,PID):"
             << "  No. of secodaries = " 
             << (*fSecondary).entries() << endl;
        G4cout << "      [Note]Secondaries from AlongStepDoIt included."
             << endl; 

        if((*fSecondary).entries()>0){
	  for(G4int lp1=0; lp1<(*fSecondary).entries(); lp1++){
               G4cout << "      "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().x() , "Length") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().y() , "Length") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().z() , "Length") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetKineticEnergy() , "Energy") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetGlobalTime() , "Time") << " "
                    << setw(18)
                    << (*fSecondary)[lp1]->GetDefinition()
                                         ->GetParticleName();
               G4cout << endl;
	     }
	}
      }

 }
/////////////////////////////////////////
void G4SteppingVerbose::StepInfo()
/////////////////////////////////////////
{
  CopyState();
  
  G4int prec = G4cout.precision(3);

  if( verboseLevel >= 1 ){
    if( verboseLevel >= 4 ) VerboseTrack();
    if( verboseLevel >= 3 ){
      G4cout << endl;
#ifdef G4_USE_G4BESTUNIT_FOR_VERBOSE      
      G4cout << setw( 5) << "#Step#" << " "
	     << setw( 8) << "X"      << "     "
	     << setw( 8) << "Y"      << "     "  
	     << setw( 8) << "Z"      << "     "
	     << setw( 9) << "KineE"  << "     "
	     << setw( 8) << "dE"     << "     "  
	     << setw(12) << "StepLeng"   << " "  
	     << setw(12) << "TrackLeng"  << " "
	     << setw(12) << "NextVolume" << " "
	     << setw( 8) << "ProcName"   << endl;	       
#else
      G4cout << setw( 5) << "#Step#"     << " "
	     << setw( 8) << "X(mm)"      << " "
	     << setw( 8) << "Y(mm)"      << " "  
	     << setw( 8) << "Z(mm)"      << " "
	     << setw( 9) << "KinE(MeV)"  << " "
	     << setw( 8) << "dE(MeV)"    << " "  
	     << setw( 8) << "StepLeng"   << " "  
	     << setw( 9) << "TrackLeng"  << " "  
	     << setw(11) << "NextVolume" << " "
	     << setw( 8) << "ProcName"   << endl;
#endif	     
    }

    G4cout << setw( 5) << fTrack->GetCurrentStepNumber() << " "
	   << setw( 8) << G4BestUnit(fTrack->GetPosition().x() , "Length") << " "
	   << setw( 8) << G4BestUnit(fTrack->GetPosition().y() , "Length") << " "
	   << setw( 8) << G4BestUnit(fTrack->GetPosition().z() , "Length") << " "
	   << setw( 9) << G4BestUnit(fTrack->GetKineticEnergy() , "Energy") << " "
	   << setw( 8) << G4BestUnit(fStep->GetTotalEnergyDeposit(), "Energy") << " "
	   << setw( 8) << G4BestUnit(fStep->GetStepLength() , "Length") << " "
	   << setw( 9) << G4BestUnit(fTrack->GetTrackLength() , "Length") << " ";

    // if( fStepStatus != fWorldBoundary){ 
    if( fTrack->GetNextVolume() != 0 ) { 
      G4cout << setw(11) << fTrack->GetNextVolume()->GetName() << " ";
    } else {
      G4cout << setw(11) << "OutOfWorld" << " ";
    }

    if(fStep->GetPostStepPoint()->GetProcessDefinedStep() != NULL){
      G4cout << fStep->GetPostStepPoint()->GetProcessDefinedStep()
	->GetProcessName();
    } else {
      G4cout << "User Limit";
    }

    G4cout << endl;

    if( verboseLevel == 2 ){
      G4int tN2ndariesTot = fN2ndariesAtRestDoIt +
	                    fN2ndariesAlongStepDoIt +
	                    fN2ndariesPostStepDoIt;
      if(tN2ndariesTot>0){
	G4cout << "    :----- List of 2ndaries - "
	       << "#SpawnInStep=" << setw(3) << tN2ndariesTot 
	       << "(Rest=" << setw(2) << fN2ndariesAtRestDoIt
	       << ",Along=" << setw(2) << fN2ndariesAlongStepDoIt
	       << ",Post="  << setw(2) << fN2ndariesPostStepDoIt
	       << "), "
	       << "#SpawnTotal=" << setw(3) << (*fSecondary).entries()
	       << " ---------------"
	       << endl;

	for(G4int lp1=(*fSecondary).entries()-tN2ndariesTot; 
                        lp1<(*fSecondary).entries(); lp1++){
	  G4cout << "    : "
		 << setw( 9)
		 << G4BestUnit((*fSecondary)[lp1]->GetPosition().x() , "Length")<< " "
		 << setw( 9)
		 << G4BestUnit((*fSecondary)[lp1]->GetPosition().y() , "Length")<< " "
		 << setw( 9)
		 << G4BestUnit((*fSecondary)[lp1]->GetPosition().z() , "Length") << " "
		 << setw( 9)
		 << G4BestUnit((*fSecondary)[lp1]->GetKineticEnergy() , "Energy")<< " "
		 << setw(18)
		 << (*fSecondary)[lp1]->GetDefinition()
 	                              ->GetParticleName();
	  G4cout << endl;
	}
              
	G4cout << "    :-----------------------------"
	       << "----------------------------------"
	       << "-- EndOf2ndaries Info ---------------"
	       << endl;
      }
    }
    
  }
  G4cout.precision(prec);
}

////////////////////////////////////////////
void G4SteppingVerbose::DPSLStarted()
////////////////////////////////////////////
{
  CopyState();

  if( verboseLevel > 5 ){
    G4cout << endl;
    G4cout << " >>DefinePhysicalStepLength (List of proposed StepLengths): "
      << endl;
  }
}
//////////////////////////////////////////////
void G4SteppingVerbose::DPSLUserLimit()
//////////////////////////////////////////////
{
  CopyState();

  if( verboseLevel > 5 ){
    G4cout << endl << endl;
    G4cout << "=== Defined Physical Step Length (DPSL)" << endl;
    G4cout << "    ++ProposedStep(UserLimit) = " 
      << setw( 9) << physIntLength
	<< " : ProcName = User defined maximum allowed Step"
	  << endl;
  }
}
/////////////////////////////////////////////
void G4SteppingVerbose::DPSLPostStep()
/////////////////////////////////////////////
{
  CopyState();

  if( verboseLevel > 5 ){
    G4cout << "    ++ProposedStep(PostStep ) = " 
      << setw( 9) << physIntLength
	<< " : ProcName = "
	  << fCurrentProcess->GetProcessName() 
            << " (";
    if(fCondition==ExclusivelyForced){
      G4cout << "ExclusivelyForced)" << endl;
    }
    else if(fCondition==Conditionally){
      G4cout << "Conditionally)" << endl;
    }
    else if(fCondition==Forced){
      G4cout << "Forced)" << endl;
    }
    else{
      G4cout << "No ForceCondition)" << endl;
    }
  }
}
/////////////////////////////////////////////
void G4SteppingVerbose::DPSLAlongStep()
/////////////////////////////////////////////
{
  CopyState();
  if( verboseLevel > 5 ){
    G4cout << "    ++ProposedStep(AlongStep) = " 
	   << setw( 9) << G4BestUnit(physIntLength , "Length")
	   << " : ProcName = "
	   << fCurrentProcess->GetProcessName() 
	   << " (";
    if(fGPILSelection==CandidateForSelection){
      G4cout << "CandidateForSelection)" << endl;
    }
    else if(fGPILSelection==NotCandidateForSelection){
      G4cout << "NotCandidateForSelection)" << endl;
    }
    else{
      G4cout << "???)" << endl;
    }
  }
}


////////////////////////////////////////////////
void G4SteppingVerbose::TrackingStarted()
////////////////////////////////////////////////
{

  CopyState();
G4int prec = G4cout.precision(3);
  if( verboseLevel > 0 ){

#ifdef G4_USE_G4BESTUNIT_FOR_VERBOSE
    G4cout << setw( 5) << "Step#"  << " "
           << setw( 8) << "X"      << "     "
	   << setw( 8) << "Y"      << "     "  
	   << setw( 8) << "Z"      << "     "
	   << setw( 9) << "KineE"  << "     "
	   << setw( 8) << "dE"     << "     "  
	   << setw(12) << "StepLeng"   << " "  
	   << setw(12) << "TrackLeng"  << " "
	   << setw(12) << "NextVolume" << " "
	   << setw( 8) << "ProcName"   << endl;	     
#else
    G4cout << setw( 5) << "Step#"      << " "
	   << setw( 8) << "X(mm)"      << " "
	   << setw( 8) << "Y(mm)"      << " "  
	   << setw( 8) << "Z(mm)"      << " "
	   << setw( 9) << "KinE(MeV)"  << " "
	   << setw( 8) << "dE(MeV)"    << " "  
	   << setw( 8) << "StepLeng"   << " "  
	   << setw( 9) << "TrackLeng"  << " "
	   << setw(11) << "NextVolume" << " "
	   << setw( 8) << "ProcName"   << endl;	     
#endif

    G4cout << setw( 5) << fTrack->GetCurrentStepNumber() << " "
	   << setw( 8) << G4BestUnit(fTrack->GetPosition().x(),"Length")<< " "
	   << setw( 8) << G4BestUnit(fTrack->GetPosition().y(),"Length") << " "
	   << setw( 8) << G4BestUnit(fTrack->GetPosition().z(),"Length")<< " "
	   << setw( 9) << G4BestUnit(fTrack->GetKineticEnergy(),"Energy")<< " "
	   << setw( 8) << G4BestUnit(fStep->GetTotalEnergyDeposit(),"Energy") << " "
	   << setw( 8) << G4BestUnit(fStep->GetStepLength(),"Length")<< " "
	   << setw( 9) << G4BestUnit(fTrack->GetTrackLength(),"Length") << " ";

    if(fTrack->GetNextVolume()){
      G4cout << setw(11) << fTrack->GetNextVolume()->GetName() << " ";
    } else {
      G4cout << setw(11) << "OutOfWorld" << " ";
    }
    G4cout << "initStep" << endl;
  }
  G4cout.precision(prec);
}
//////////////////////////////////////////////////////
void G4SteppingVerbose::AlongStepDoItOneByOne()
//////////////////////////////////////////////////////
{ 
  CopyState();
    if(verboseLevel >= 4){ 
        G4cout << endl;
        G4cout << " >>AlongStepDoIt (process by process): "
             << "   Process Name = " 
             << fCurrentProcess->GetProcessName() << endl;

        ShowStep();
        G4cout << "          "
	       << "!Note! Safety of PostStep is only valid "
	       << "after all DoIt invocations."
	       << endl; 

        VerboseParticleChange();    
        G4cout << endl;

        G4cout << "    ++List of secondaries generated " 
	       << "(x,y,z,kE,t,PID):"
	       << "  No. of secodaries = " 
	       << fN2ndariesAlongStepDoIt << endl;

        if(fN2ndariesAlongStepDoIt>0){
           for(G4int lp1=(*fSecondary).entries()-fN2ndariesAlongStepDoIt; 
                     lp1<(*fSecondary).entries(); lp1++){
               G4cout << "      "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().x() , "Length")<< " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().y() , "Length")<< " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().z() , "Length")<< " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetKineticEnergy() , "Energy")<< " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetGlobalTime() , "Time")<< " "
                    << setw(18)
                    << (*fSecondary)[lp1]->GetDefinition()
                                         ->GetParticleName();
               G4cout << endl;
	   }
	}
     }

}
//////////////////////////////////////////////////////
void G4SteppingVerbose::PostStepDoItOneByOne()
//////////////////////////////////////////////////////
{
  CopyState();
     if(fStepStatus != fPostStepDoItProc) return;

     if(verboseLevel >= 4){ 
        G4cout << endl;
        G4cout << " >>PostStepDoIt (process by process): "
             << "   Process Name = " 
             << fCurrentProcess->GetProcessName() << endl;

        ShowStep();
        G4cout << endl;
        VerboseParticleChange();    
        G4cout << endl;
         
        G4cout << "    ++List of secondaries generated " 
             << "(x,y,z,kE,t,PID):"
             << "  No. of secodaries = " 
             << fN2ndariesPostStepDoIt << endl;

        if(fN2ndariesPostStepDoIt>0){
           for(G4int lp1=(*fSecondary).entries()-fN2ndariesPostStepDoIt; 
                     lp1<(*fSecondary).entries(); lp1++){
               G4cout << "      "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().x() , "Length")<< " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().y(), "Length") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetPosition().z(), "Length") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetKineticEnergy(), "Energy") << " "
                    << setw( 9)
                    << G4BestUnit((*fSecondary)[lp1]->GetGlobalTime(), "Time") << " "
                    << setw(18)
                    << (*fSecondary)[lp1]->GetDefinition()
                                         ->GetParticleName();
               G4cout << endl;
	   }
	}
     }

}


//////////////////////////////////////
void G4SteppingVerbose::VerboseTrack()
//////////////////////////////////////
{
  CopyState();
// Show header
  G4cout << endl;
  G4cout << "    ++G4Track Information " << endl;
  G4int prec = G4cout.precision(3);


  G4cout << "      -----------------------------------------------" 
       << endl;
  G4cout << "        G4Track Information  " << setw(20) << endl;
  G4cout << "      -----------------------------------------------" 
       << endl;

  G4cout << "        Step number         : " 
       << setw(20) << fTrack->GetCurrentStepNumber()
       << endl; 
#ifdef G4_USE_G4BESTUNIT_FOR_VERBOSE
  G4cout << "        Position - x        : " 
       << setw(20) << G4BestUnit(fTrack->GetPosition().x(), "Length")
       << endl; 
  G4cout << "        Position - y        : " 
       << setw(20) << G4BestUnit(fTrack->GetPosition().y(), "Length")
       << endl; 
  G4cout << "        Position - z        : " 
       << setw(20) << G4BestUnit(fTrack->GetPosition().z(), "Length")
       << endl;
  G4cout << "        Global Time         : " 
       << setw(20) << G4BestUnit(fTrack->GetGlobalTime(), "Time")
       << endl;
  G4cout << "        Local Time          : " 
       << setw(20) << G4BestUnit(fTrack->GetLocalTime(), "Time")
       << endl;
#else
  G4cout << "        Position - x (mm)   : " 
       << setw(20) << fTrack->GetPosition().x() /mm
       << endl; 
  G4cout << "        Position - y (mm)   : " 
       << setw(20) << fTrack->GetPosition().y() /mm
       << endl; 
  G4cout << "        Position - z (mm)   : " 
       << setw(20) << fTrack->GetPosition().z() /mm
       << endl;
  G4cout << "        Global Time (ns)    : " 
       << setw(20) << fTrack->GetGlobalTime() /ns
       << endl;
  G4cout << "        Local Time (ns)     : " 
       << setw(20) << fTrack->GetLocalTime() /ns
       << endl;
#endif
  G4cout << "        Momentum Direct - x : " 
       << setw(20) << fTrack->GetMomentumDirection().x()
       << endl;
  G4cout << "        Momentum Direct - y : " 
       << setw(20) << fTrack->GetMomentumDirection().y()
       << endl;
  G4cout << "        Momentum Direct - z : " 
       << setw(20) << fTrack->GetMomentumDirection().z()
       << endl;
#ifdef G4_USE_G4BESTUNIT_FOR_VERBOSE
  G4cout << "        Kinetic Energy      : " 
#else
  G4cout << "        Kinetic Energy (MeV): " 
#endif
       << setw(20) << G4BestUnit(fTrack->GetKineticEnergy(), "Energy")
       << endl;
  G4cout << "        Polarization - x    : " 
       << setw(20) << fTrack->GetPolarization().x()
       << endl;
  G4cout << "        Polarization - y    : " 
       << setw(20) << fTrack->GetPolarization().y()
       << endl;
  G4cout << "        Polarization - z    : " 
       << setw(20) << fTrack->GetPolarization().z()
       << endl;
  G4cout << "        Track Length        : " 
       << setw(20) << G4BestUnit(fTrack->GetTrackLength(), "Length")
       << endl;
  G4cout << "        Track ID #          : " 
       << setw(20) << fTrack->GetTrackID()
       << endl;
  G4cout << "        Parent Track ID #   : " 
       << setw(20) << fTrack->GetParentID()
       << endl;
  G4cout << "        Next Volume         : " 
       << setw(20);
       if( fTrack->GetNextVolume() != 0 ) { 
         G4cout << fTrack->GetNextVolume()->GetName() << " ";
       } else {
         G4cout << "OutOfWorld" << " ";
       }
       G4cout << endl;
  G4cout << "        Track Status        : " 
       << setw(20);
       if( fTrack->GetTrackStatus() == fAlive ){
         G4cout << " Alive";
       } else if( fTrack->GetTrackStatus() == fStopButAlive ){
           G4cout << " StopButAlive";
       } else if( fTrack->GetTrackStatus() == fStopAndKill ){
           G4cout << " StopAndKill";
       } else if( fTrack->GetTrackStatus() == fKillTrackAndSecondaries ){
           G4cout << " KillTrackAndSecondaries";
       } else if( fTrack->GetTrackStatus() == fSuspend ){
           G4cout << " Suspend";
       } else if( fTrack->GetTrackStatus() == fPostponeToNextEvent ){
           G4cout << " PostponeToNextEvent";
       }
       G4cout << endl;
#ifdef G4_USE_G4BESTUNIT_FOR_VERBOSE
  G4cout << "        Vertex - x          : " 
       << setw(20) << G4BestUnit(fTrack->GetVertexPosition().x(),"Length")
       << endl; 
  G4cout << "        Vertex - y          : " 
       << setw(20) << G4BestUnit(fTrack->GetVertexPosition().y(),"Length")
       << endl; 
  G4cout << "        Vertex - z          : " 
       << setw(20) << G4BestUnit(fTrack->GetVertexPosition().z(),"Length")
       << endl;
#else
  G4cout << "        Vertex - x (mm)     : " 
       << setw(20) << fTrack->GetVertexPosition().x()/mm
       << endl; 
  G4cout << "        Vertex - y (mm)     : " 
       << setw(20) << fTrack->GetVertexPosition().y()/mm
       << endl; 
  G4cout << "        Vertex - z (mm)     : " 
       << setw(20) << fTrack->GetVertexPosition().z()/mm
       << endl;
#endif
  G4cout << "        Vertex - Px (MomDir): " 
       << setw(20) << fTrack->GetVertexMomentumDirection().x()
       << endl;
  G4cout << "        Vertex - Py (MomDir): " 
       << setw(20) << fTrack->GetVertexMomentumDirection().y()
       << endl;
  G4cout << "        Vertex - Pz (MomDir): " 
       << setw(20) << fTrack->GetVertexMomentumDirection().z()
       << endl;
#ifdef G4_USE_G4BESTUNIT_FOR_VERBOSE
  G4cout << "        Vertex - KineE      : " 
#else
  G4cout << "        Vertex - KineE (MeV): " 
#endif
       << setw(20) << G4BestUnit(fTrack->GetVertexKineticEnergy(),"Energy")
       << endl;
  
  G4cout << "        Creator Process     : " 
       << setw(20);
  if( fTrack->GetCreatorProcess() == NULL){
    G4cout << " Event Generator" << endl;
  } else {
    G4cout << fTrack->GetCreatorProcess()->GetProcessName() << endl;
  }

  G4cout << "      -----------------------------------------------" 
       << endl;
       
 G4cout.precision(prec);      
}


///////////////////////////////////////////////
void G4SteppingVerbose::VerboseParticleChange()
///////////////////////////////////////////////
{

// Show header
  G4cout << endl;
  G4cout << "    ++G4ParticleChange Information " << endl;
  fParticleChange->DumpInfo();
}
/////////////////////////////////////////
void G4SteppingVerbose::ShowStep() const
////////////////////////////////////////
{

// Show header
   G4cout << endl;
   G4cout << "    ++G4Step Information " << endl;
   G4cout.precision(3);

// Show G4Step specific information
   G4cout << "      Address of G4Track    : " << fStep->GetTrack() << endl;
   G4cout << "      Step Length (mm)      : " << fStep->GetTrack()->GetStepLength() << endl;
   G4cout << "      Energy Deposit (MeV)  : " << fStep->GetTotalEnergyDeposit() << endl;

// Show G4StepPoint specific information
   G4cout << "      -------------------------------------------------------" 
        << "----------------" <<  endl;
   G4cout << "        StepPoint Information  " << setw(20) << "PreStep" 
                                             << setw(20) << "PostStep" << endl;
   G4cout << "      -------------------------------------------------------" 
        << "----------------" <<  endl;
   G4cout << "         Position - x (mm)   : " 
        << setw(20) << fStep->GetPreStepPoint()->GetPosition().x() 
        << setw(20) << fStep->GetPostStepPoint()->GetPosition().x() << endl;
   G4cout << "         Position - y (mm)   : " 
        << setw(20) << fStep->GetPreStepPoint()->GetPosition().y() 
        << setw(20) << fStep->GetPostStepPoint()->GetPosition().y() << endl;
   G4cout << "         Position - z (mm)   : " 
        << setw(20) << fStep->GetPreStepPoint()->GetPosition().z() 
        << setw(20) << fStep->GetPostStepPoint()->GetPosition().z() << endl;
   G4cout << "         Global Time (ns)    : " 
        << setw(20) << fStep->GetPreStepPoint()->GetGlobalTime()
        << setw(20) << fStep->GetPostStepPoint()->GetGlobalTime() << endl;
   G4cout << "         Local Time (ns)     : " 
        << setw(20) << fStep->GetPreStepPoint()->GetLocalTime() 
        << setw(20) << fStep->GetPostStepPoint()->GetLocalTime() << endl;
   G4cout << "         Proper Time (ns)    : " 
        << setw(20) << fStep->GetPreStepPoint()->GetProperTime()
        << setw(20) << fStep->GetPostStepPoint()->GetProperTime() << endl;
   G4cout << "         Momentum Direct - x : " 
        << setw(20) << fStep->GetPreStepPoint()->GetMomentumDirection().x()
        << setw(20) << fStep->GetPostStepPoint()->GetMomentumDirection().x() << endl;
   G4cout << "         Momentum Direct - y : " 
        << setw(20) << fStep->GetPreStepPoint()->GetMomentumDirection().y()
        << setw(20) << fStep->GetPostStepPoint()->GetMomentumDirection().y() << endl;
   G4cout << "         Momentum Direct - z : " 
        << setw(20) << fStep->GetPreStepPoint()->GetMomentumDirection().z()
        << setw(20) << fStep->GetPostStepPoint()->GetMomentumDirection().z() << endl;
   G4cout << "         Momentum - x (MeV/c): " 
        << setw(20) << fStep->GetPreStepPoint()->GetMomentum().x()
        << setw(20) << fStep->GetPostStepPoint()->GetMomentum().x() << endl;
   G4cout << "         Momentum - y (MeV/c): " 
        << setw(20) << fStep->GetPreStepPoint()->GetMomentum().y()
        << setw(20) << fStep->GetPostStepPoint()->GetMomentum().y() << endl;
   G4cout << "         Momentum - z (MeV/c): " 
        << setw(20) << fStep->GetPreStepPoint()->GetMomentum().z()
        << setw(20) << fStep->GetPostStepPoint()->GetMomentum().z() << endl;
   G4cout << "         Total Energy (MeV)  : " 
        << setw(20) << fStep->GetPreStepPoint()->GetTotalEnergy()
        << setw(20) << fStep->GetPostStepPoint()->GetTotalEnergy() << endl;
   G4cout << "         Kinetic Energy (MeV): " 
        << setw(20) << fStep->GetPreStepPoint()->GetKineticEnergy()
        << setw(20) << fStep->GetPostStepPoint()->GetKineticEnergy() << endl;
   G4cout << "         Velocity (mm/ns)    : " 
        << setw(20) << fStep->GetPreStepPoint()->GetVelocity()
        << setw(20) << fStep->GetPostStepPoint()->GetVelocity() << endl;
   G4cout << "         Volume Name         : " 
        << setw(20) << fStep->GetPreStepPoint()->GetPhysicalVolume()->GetName()
        << setw(20) << fStep->GetPostStepPoint()->GetPhysicalVolume()->GetName() << endl;
   G4cout << "         Safety (mm)         : " 
        << setw(20) << fStep->GetPreStepPoint()->GetSafety()
        << setw(20) << fStep->GetPostStepPoint()->GetSafety() << endl;
   G4cout << "         Polarization - x    : " 
        << setw(20) << fStep->GetPreStepPoint()->GetPolarization().x()
        << setw(20) << fStep->GetPostStepPoint()->GetPolarization().x() << endl;
   G4cout << "         Polarization - y    : " 
        << setw(20) << fStep->GetPreStepPoint()->GetPolarization().y()
        << setw(20) << fStep->GetPostStepPoint()->GetPolarization().y() << endl;
   G4cout << "         Polarization - Z    : " 
        << setw(20) << fStep->GetPreStepPoint()->GetPolarization().z()
        << setw(20) << fStep->GetPostStepPoint()->GetPolarization().z() << endl;
   G4cout << "         Weight              : " 
        << setw(20) << fStep->GetPreStepPoint()->GetWeight()
        << setw(20) << fStep->GetPostStepPoint()->GetWeight() << endl;
   G4cout << "         Step Status         : " ;
        G4StepStatus  tStepStatus = fStep->GetPreStepPoint()->GetStepStatus();
        if( tStepStatus == fGeomBoundary ){
           G4cout << setw(20) << "Geom Limit";
        } else if ( tStepStatus == fAlongStepDoItProc ){
          G4cout << setw(20) << "AlongStep Proc.";
        } else if ( tStepStatus == fPostStepDoItProc ){
           G4cout << setw(20) << "PostStep Proc";
        } else if ( tStepStatus == fAtRestDoItProc ){
           G4cout << setw(20) << "AtRest Proc";
        } else if ( tStepStatus == fUndefined ){
           G4cout << setw(20) << "Undefined";
        }

        tStepStatus = fStep->GetPostStepPoint()->GetStepStatus();
        if( tStepStatus == fGeomBoundary ){
           G4cout << setw(20) << "Geom Limit";
        } else if ( tStepStatus == fAlongStepDoItProc ){
           G4cout << setw(20) << "AlongStep Proc.";
        } else if ( tStepStatus == fPostStepDoItProc ){
           G4cout << setw(20) << "PostStep Proc";
        } else if ( tStepStatus == fAtRestDoItProc ){
           G4cout << setw(20) << "AtRest Proc";
        } else if ( tStepStatus == fUndefined ){
           G4cout << setw(20) << "Undefined";
        }

        G4cout << endl;
        G4cout << "         Process defined Step: " ;
        if( fStep->GetPreStepPoint()->GetProcessDefinedStep() == NULL ){
 	   G4cout << setw(20) << "Undefined";
        } else {
  	   G4cout << setw(20) << fStep->GetPreStepPoint()->GetProcessDefinedStep()
                                             ->GetProcessName();
        }
        if( fStep->GetPostStepPoint()->GetProcessDefinedStep() == NULL){
  	   G4cout << setw(20) << "Undefined";
        } else {
 	   G4cout << setw(20) << fStep->GetPostStepPoint()->GetProcessDefinedStep()
                                              ->GetProcessName(); 
        }

   G4cout << endl;
   G4cout << "      -------------------------------------------------------" 
        << "----------------" << endl;
}


