/*****************************************************************************\
* (c) Copyright 2000-2020 CERN for the benefit of the LHCb Collaboration      *
*                                                                             *
* This software is distributed under the terms of the GNU General Public      *
* Licence version 3 (GPL Version 3), copied verbatim in the file "COPYING".   *
*                                                                             *
* In applying this licence, CERN does not waive the privileges and immunities *
* granted to it by virtue of its status as an Intergovernmental Organization  *
* or submit itself to any jurisdiction.                                       *
\*****************************************************************************/
// $Id: TupleToolCentrality.cpp,v 1.0 2020-11-12 garcia $
// Include files

// local
#include "TupleToolCentrality.h"

#include "GaudiAlg/ITupleTool.h"
#include "GaudiAlg/Tuple.h"

#include "Event/CaloDigit.h"

#include "GaudiKernel/IRegistry.h" // IOpaqueAddress

#include "CaloDet/DeCalorimeter.h"

#include <math.h>

//-----------------------------------------------------------------------------
// Implementation file for class : TupleToolCentrality
//
// Based on TupleToolCaloDigits by 2015-08-05 Albert Bursche
//-----------------------------------------------------------------------------

// Declaration of the Tool Factory
DECLARE_COMPONENT( TupleToolCentrality )

//=============================================================================
// Standard constructor, initializes variables
//=============================================================================
TupleToolCentrality::TupleToolCentrality( const std::string& type, const std::string& name, const IInterface* parent )
    : TupleToolBase( type, name, parent ) {
  declareInterface<IEventTupleTool>( this );
}
//=============================================================================

StatusCode TupleToolCentrality::initialize() {
  if ( 100 % m_nclasses == 0 ) {

    if ( m_nclasses == 1 ) {
      msg() << "TupleToolCentrality requested to use 1 centrality class" << endmsg;
    } else {
      msg() << Form( "TupleToolCentrality requested to use %i centrality classes", m_nclasses.value() ) << endmsg;
    }

    ////// Width of classes distributions  ///

    std::for_each( m_CollSyst.begin(), m_CollSyst.end(),
                   []( char& c ) { c = ::tolower( c ); } ); // make collision system lowercase => case insensitive.

    if ( m_CollSyst == "pbpb" ) {
      msg() << "TupleToolCentrality requested PbPb information" << endmsg;
      m_ecalcuts = {
          0.0,        70000.0,    100000.0,   120000.0,   150000.0,   170000.0,   200000.0,   220000.0,   250000.0,
          280000.0,   310000.0,   350000.0,   380000.0,   420000.0,   470000.0,   510000.0,   560000.0,   620000.0,
          670000.0,   730000.0,   800000.0,   870000.0,   950000.0,   1030000.0,  1110000.0,  1200000.0,  1300000.0,
          1400000.0,  1510000.0,  1630000.0,  1750000.0,  1880000.0,  2010000.0,  2150000.0,  2300000.0,  2460000.0,
          2620000.0,  2800000.0,  2980000.0,  3160000.0,  3360000.0,  3570000.0,  3790000.0,  4020000.0,  4250000.0,
          4500000.0,  4750000.0,  5020000.0,  5300000.0,  5590000.0,  5900000.0,  6200000.0,  6530000.0,  6870000.0,
          7230000.0,  7600000.0,  7980000.0,  8370000.0,  8780000.0,  9190000.0,  9630000.0,  10090000.0, 10550000.0,
          11020000.0, 11510000.0, 12020000.0, 12550000.0, 13090000.0, 13660000.0, 14260000.0, 14860000.0, 15480000.0,
          16130000.0, 16790000.0, 17470000.0, 18200000.0, 18940000.0, 19700000.0, 20480000.0, 21290000.0, 22150000.0,
          23010000.0, 23910000.0, 24820000.0, 25760000.0, 26750000.0, 27770000.0, 28840000.0, 29950000.0, 31100000.0,
          32280000.0, 33530000.0, 34810000.0, 36130000.0, 37500000.0, 38930000.0, 40440000.0, 41990000.0, 43610000.0,
          45410000.0}; // values taken for 100 classes and 6000 bins on ecal glauber from 0 to 60E6 MeV for PbPb at
                       // √s_nn = 5.02 TeV.

      m_npart = {2.13,   2.27,   2.39,   2.55,   2.70,   2.90,   3.16,   3.42,   3.73,   4.04,   4.52,   4.93,   5.42,
                 5.99,   6.56,   7.17,   7.83,   8.57,   9.28,   10.06,  10.99,  11.91,  12.93,  13.89,  15.00,  16.24,
                 17.48,  18.74,  20.21,  21.61,  23.17,  24.76,  26.36,  28.13,  29.92,  31.95,  33.89,  35.98,  38.04,
                 40.28,  42.64,  45.07,  47.56,  50.26,  52.77,  55.62,  58.49,  61.47,  64.59,  67.79,  70.90,  74.29,
                 77.87,  81.50,  85.20,  89.05,  92.87,  97.01,  101.06, 105.25, 109.80, 114.23, 118.69, 123.32, 128.35,
                 133.12, 138.15, 143.46, 148.78, 154.50, 160.13, 165.71, 171.71, 177.58, 184.03, 190.61, 197.33, 204.03,
                 210.87, 218.09, 225.35, 232.93, 240.84, 248.64, 256.65, 265.01, 273.62, 282.39, 291.69, 301.06, 310.44,
                 320.74, 330.98, 341.21, 351.91, 362.83, 373.72, 384.38, 393.79, 401.71};

      m_npart_sys = {0.491, 0.591, 0.782, 0.793, 0.568, 0.697, 0.509, 0.812, 1.14,  1.113,
                     1.204, 0.776, 1.332, 0.938, 1.571, 1.399, 1.85,  1.801, 0.448, 2.356};

      m_ncoll = {1.13,    1.26,    1.36,    1.51,    1.64,    1.83,    2.06,    2.29,    2.55,    2.84,
                 3.26,    3.65,    4.12,    4.67,    5.25,    5.84,    6.51,    7.33,    8.11,    8.94,
                 10.05,   11.10,   12.34,   13.59,   14.98,   16.70,   18.34,   20.20,   22.36,   24.50,
                 26.86,   29.49,   32.27,   35.24,   38.36,   42.10,   45.78,   49.86,   54.02,   58.61,
                 63.75,   69.05,   74.85,   81.13,   87.23,   94.15,   101.60,  109.39,  117.56,  126.61,
                 135.82,  145.38,  155.83,  167.14,  178.91,  191.39,  204.06,  217.25,  231.95,  246.55,
                 262.14,  279.10,  295.46,  312.78,  331.88,  351.75,  371.87,  394.13,  415.85,  440.58,
                 464.59,  490.20,  516.64,  543.35,  572.94,  603.81,  636.28,  669.16,  702.61,  738.93,
                 775.43,  814.83,  855.98,  897.52,  939.72,  987.79,  1034.64, 1084.22, 1139.00, 1194.86,
                 1250.94, 1313.38, 1376.87, 1444.74, 1517.07, 1590.93, 1672.70, 1756.68, 1843.50, 1936.92};

      m_ncoll_sys = {0.291, 0.392, 0.594, 0.676, 0.553, 0.81,  0.673, 1.144, 1.895,  2.134,
                     2.603, 2.25,  3.91,  3.851, 6.222, 6.981, 9.463, 11.19, 11.834, 18.519};

      m_b = {15.64, 15.62, 15.56, 15.52, 15.47, 15.40, 15.33, 15.24, 15.18, 15.09, 15.00, 14.89, 14.82, 14.70, 14.60,
             14.52, 14.41, 14.32, 14.22, 14.15, 14.03, 13.93, 13.84, 13.75, 13.64, 13.54, 13.45, 13.35, 13.25, 13.16,
             13.06, 12.96, 12.87, 12.77, 12.66, 12.56, 12.46, 12.37, 12.27, 12.17, 12.06, 11.96, 11.85, 11.75, 11.65,
             11.54, 11.43, 11.32, 11.21, 11.10, 10.99, 10.88, 10.77, 10.65, 10.53, 10.41, 10.30, 10.18, 10.06, 9.94,
             9.80,  9.68,  9.55,  9.44,  9.30,  9.17,  9.03,  8.91,  8.77,  8.62,  8.48,  8.34,  8.19,  8.04,  7.89,
             7.73,  7.57,  7.41,  7.24,  7.07,  6.90,  6.72,  6.53,  6.34,  6.16,  5.95,  5.74,  5.53,  5.30,  5.06,
             4.82,  4.55,  4.27,  3.98,  3.65,  3.29,  2.89,  2.42,  1.90,  1.35};

      m_b_sys = {3.02,  2.91,  2.18, 1.32,  0.59,  0.44,  0.25,  0.3,  0.26, 0.21,
                 0.181, 0.081, 0.1,  0.062, 0.071, 0.051, 0.051, 0.04, 0.01, 0.024};

      if ( m_nclasses == 100 ) {
        m_npart_stat = {0.004, 0.006, 0.007, 0.008, 0.010, 0.011, 0.013, 0.014, 0.015, 0.016, 0.017, 0.018, 0.019,
                        0.021, 0.021, 0.022, 0.023, 0.025, 0.026, 0.026, 0.028, 0.029, 0.030, 0.031, 0.032, 0.033,
                        0.035, 0.036, 0.037, 0.038, 0.040, 0.041, 0.042, 0.043, 0.044, 0.046, 0.047, 0.049, 0.050,
                        0.052, 0.052, 0.054, 0.056, 0.057, 0.057, 0.059, 0.061, 0.063, 0.064, 0.066, 0.067, 0.068,
                        0.071, 0.072, 0.073, 0.075, 0.077, 0.079, 0.080, 0.081, 0.084, 0.085, 0.086, 0.088, 0.090,
                        0.092, 0.095, 0.095, 0.096, 0.099, 0.100, 0.102, 0.104, 0.106, 0.108, 0.110, 0.112, 0.113,
                        0.114, 0.116, 0.117, 0.119, 0.121, 0.122, 0.123, 0.125, 0.126, 0.128, 0.130, 0.129, 0.130,
                        0.133, 0.132, 0.133, 0.133, 0.131, 0.130, 0.121, 0.105, 0.077};
        m_ncoll_stat = {0.004, 0.005, 0.007, 0.008, 0.009, 0.010, 0.012, 0.013, 0.014, 0.015, 0.017, 0.018, 0.020,
                        0.022, 0.023, 0.025, 0.027, 0.029, 0.031, 0.033, 0.036, 0.039, 0.041, 0.045, 0.047, 0.052,
                        0.057, 0.061, 0.067, 0.071, 0.075, 0.081, 0.088, 0.094, 0.100, 0.107, 0.114, 0.123, 0.130,
                        0.137, 0.146, 0.153, 0.166, 0.173, 0.182, 0.193, 0.203, 0.212, 0.223, 0.235, 0.246, 0.260,
                        0.272, 0.280, 0.293, 0.307, 0.323, 0.332, 0.346, 0.357, 0.371, 0.383, 0.397, 0.411, 0.434,
                        0.442, 0.454, 0.472, 0.490, 0.496, 0.514, 0.524, 0.545, 0.555, 0.570, 0.586, 0.608, 0.622,
                        0.627, 0.640, 0.658, 0.668, 0.688, 0.708, 0.721, 0.737, 0.751, 0.756, 0.786, 0.800, 0.812,
                        0.836, 0.854, 0.876, 0.902, 0.920, 0.960, 0.979, 0.967, 0.952};
        m_b_stat     = {0.011, 0.010, 0.010, 0.010, 0.010, 0.010, 0.010, 0.010, 0.010, 0.010, 0.009, 0.009, 0.009,
                    0.009, 0.009, 0.008, 0.008, 0.008, 0.008, 0.007, 0.007, 0.007, 0.007, 0.007, 0.006, 0.006,
                    0.006, 0.006, 0.006, 0.006, 0.006, 0.005, 0.005, 0.005, 0.005, 0.005, 0.005, 0.005, 0.005,
                    0.005, 0.005, 0.005, 0.005, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004,
                    0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.003, 0.003, 0.003,
                    0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003,
                    0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003,
                    0.004, 0.004, 0.004, 0.004, 0.004, 0.005, 0.006, 0.006, 0.006};
      } else if ( m_nclasses == 50 ) {
        m_npart_stat = {0.004, 0.006, 0.007, 0.009, 0.011, 0.013, 0.014, 0.016, 0.017, 0.019, 0.020, 0.022, 0.023,
                        0.025, 0.027, 0.029, 0.031, 0.033, 0.035, 0.037, 0.039, 0.041, 0.042, 0.045, 0.047, 0.049,
                        0.052, 0.054, 0.057, 0.059, 0.062, 0.064, 0.067, 0.070, 0.072, 0.074, 0.077, 0.080, 0.083,
                        0.085, 0.088, 0.090, 0.092, 0.095, 0.097, 0.100, 0.101, 0.101, 0.096, 0.071};
        m_ncoll_stat = {0.003, 0.005, 0.007, 0.009, 0.010, 0.012, 0.015, 0.017, 0.020, 0.023, 0.027, 0.031, 0.036,
                        0.042, 0.049, 0.056, 0.065, 0.074, 0.085, 0.096, 0.107, 0.122, 0.135, 0.149, 0.165, 0.182,
                        0.199, 0.217, 0.236, 0.254, 0.273, 0.292, 0.318, 0.337, 0.359, 0.378, 0.400, 0.423, 0.450,
                        0.466, 0.489, 0.515, 0.543, 0.561, 0.595, 0.623, 0.657, 0.695, 0.747, 0.755};
        m_b_stat     = {0.007, 0.007, 0.007, 0.007, 0.007, 0.007, 0.006, 0.006, 0.006, 0.005, 0.005, 0.005, 0.004,
                    0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003,
                    0.003, 0.003, 0.003, 0.003, 0.003, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002,
                    0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.003, 0.003, 0.003, 0.004, 0.005};
      } else if ( m_nclasses == 25 ) {
        m_npart_stat = {0.003, 0.006, 0.009, 0.011, 0.013, 0.016, 0.018, 0.021, 0.024, 0.027, 0.031, 0.034, 0.038,
                        0.042, 0.046, 0.050, 0.054, 0.059, 0.064, 0.069, 0.074, 0.079, 0.085, 0.089, 0.076};
        m_ncoll_stat = {0.003, 0.006, 0.008, 0.012, 0.016, 0.021, 0.029, 0.039, 0.052, 0.068, 0.086, 0.107, 0.131,
                        0.159, 0.188, 0.217, 0.253, 0.288, 0.326, 0.366, 0.409, 0.457, 0.518, 0.598, 0.688};
        m_b_stat     = {0.005, 0.005, 0.005, 0.004, 0.004, 0.003, 0.003, 0.003, 0.003, 0.003, 0.002, 0.002, 0.002,
                    0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.003, 0.004};
      } else if ( m_nclasses == 20 ) {
        m_npart_stat = {0.003, 0.006, 0.009, 0.012, 0.015, 0.018, 0.022, 0.025, 0.030, 0.034,
                        0.039, 0.043, 0.049, 0.054, 0.060, 0.067, 0.073, 0.081, 0.088, 0.080};
        m_ncoll_stat = {0.003, 0.006, 0.009, 0.014, 0.020, 0.030, 0.043, 0.061, 0.083, 0.108,
                        0.139, 0.173, 0.210, 0.253, 0.297, 0.349, 0.403, 0.475, 0.568, 0.693};
        m_b_stat     = {0.005, 0.004, 0.004, 0.004, 0.003, 0.003, 0.002, 0.002, 0.002, 0.002,
                    0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.004};
      } else if ( m_nclasses == 10 ) {
        m_npart_stat = {0.004, 0.009, 0.015, 0.022, 0.032, 0.042, 0.053, 0.068, 0.086, 0.102};
        m_ncoll_stat = {0.004, 0.010, 0.022, 0.047, 0.088, 0.148, 0.226, 0.333, 0.480, 0.749};
        m_b_stat     = {0.003, 0.003, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.004};
      } else if ( m_nclasses == 5 ) {
        m_npart_stat = {0.007, 0.022, 0.045, 0.077, 0.126};
        m_ncoll_stat = {0.007, 0.038, 0.136, 0.343, 0.803};
        m_b_stat     = {0.002, 0.002, 0.002, 0.002, 0.004};
      } else if ( m_nclasses == 4 ) {
        m_npart_stat = {0.009, 0.033, 0.070, 0.132};
        m_ncoll_stat = {0.010, 0.071, 0.275, 0.805};
        m_b_stat     = {0.002, 0.002, 0.002, 0.004};
      } else if ( m_nclasses == 2 ) {
        m_npart_stat = {0.028, 0.139};
        m_ncoll_stat = {0.052, 0.727};
        m_b_stat     = {0.002, 0.004};
      } else if ( m_nclasses == 1 ) {
        m_npart_stat = {0.116};
        m_ncoll_stat = {0.506};
        m_b_stat     = {0.004};
      }
    } else if ( m_CollSyst == "pbne" ) {
      msg() << "TupleToolCentrality requested PbNe information" << endmsg;
      m_ecalcuts = {0.0,       25000.0,   35833.3,   45000.0,   52500.0,   60000.0,   66666.7,   73333.3,   80000.0,
                    86666.7,   93333.3,   100833.3,  108333.3,  116666.7,  125000.0,  133333.3,  142500.0,  152500.0,
                    162500.0,  172500.0,  183333.3,  195000.0,  206666.7,  218333.3,  231666.7,  244166.7,  258333.3,
                    273333.3,  288333.3,  305000.0,  321666.7,  339166.7,  356666.7,  375833.3,  395000.0,  415833.3,
                    436666.7,  458333.3,  481666.7,  505000.0,  530000.0,  555000.0,  580833.3,  608333.3,  635833.3,
                    665000.0,  695000.0,  725833.3,  756666.7,  790000.0,  823333.3,  858333.3,  893333.3,  929166.7,
                    966666.7,  1005000.0, 1044166.7, 1083333.4, 1125000.0, 1165833.4, 1208333.4, 1250833.4, 1294166.6,
                    1340833.4, 1386666.6, 1435000.0, 1482500.0, 1531666.6, 1581666.6, 1631666.6, 1683333.4, 1736666.6,
                    1789166.6, 1841666.6, 1897500.0, 1952500.0, 2008333.4, 2065833.4, 2124166.8, 2181666.8, 2241666.8,
                    2300000.0, 2360833.2, 2420833.2, 2482500.0, 2543333.2, 2605000.0, 2669166.8, 2733333.2, 2798333.2,
                    2865000.0, 2934166.8, 3006666.8, 3081666.8, 3162500.0, 3250000.0, 3350000.0, 3465833.2, 3611666.8,
                    3824166.8}; // values taken for 100 classes and 6000 bins on ecal glauber from 0 to 5E6 MeV for PbNe
                                // at s_nn = 69 GeV.
      m_npart     = {2.09,  2.18,  2.25,  2.32,  2.39,  2.45,  2.53,  2.61,  2.73,  2.82,  3.00,  3.17,  3.34,
                 3.57,  3.75,  3.94,  4.18,  4.41,  4.63,  4.87,  5.20,  5.48,  5.78,  6.13,  6.48,  6.87,
                 7.19,  7.60,  8.02,  8.46,  8.92,  9.36,  9.86,  10.34, 10.91, 11.45, 11.94, 12.59, 13.17,
                 13.79, 14.40, 15.10, 15.80, 16.56, 17.30, 18.03, 18.79, 19.63, 20.48, 21.28, 22.26, 23.12,
                 24.08, 24.99, 25.99, 27.02, 28.00, 29.06, 30.10, 31.21, 32.33, 33.41, 34.64, 35.90, 37.02,
                 38.36, 39.71, 40.89, 42.18, 43.66, 45.11, 46.39, 47.74, 49.24, 50.71, 52.26, 53.63, 55.32,
                 56.78, 58.31, 59.89, 61.60, 63.10, 64.93, 66.28, 67.97, 69.48, 71.25, 72.61, 74.41, 76.03,
                 77.44, 79.17, 80.90, 82.60, 84.41, 86.57, 88.91, 92.00, 97.76};
      m_npart_sys = {0.092, 0.05,  0.164, 0.135, 0.225, 0.279, 0.304, 0.24, 0.204, 0.282,
                     0.247, 0.353, 0.521, 0.576, 0.519, 0.617, 0.772, 0.8,  0.873, 1.082};
      m_ncoll = {1.08,   1.16,   1.23,   1.30,   1.36,   1.41,   1.48,   1.54,   1.65,  1.72,   1.87,   2.03,   2.17,
                 2.36,   2.51,   2.67,   2.85,   3.06,   3.25,   3.45,   3.73,   3.98,  4.27,   4.56,   4.88,   5.23,
                 5.55,   5.93,   6.34,   6.73,   7.19,   7.63,   8.11,   8.65,   9.24,  9.81,   10.32,  11.00,  11.63,
                 12.35,  13.01,  13.87,  14.66,  15.55,  16.41,  17.33,  18.30,  19.37, 20.45,  21.43,  22.72,  23.91,
                 25.15,  26.42,  27.77,  29.18,  30.63,  32.13,  33.63,  35.17,  36.89, 38.50,  40.42,  42.28,  43.87,
                 46.06,  48.27,  50.03,  52.21,  54.41,  56.78,  58.56,  61.01,  63.50, 65.82,  68.67,  70.89,  73.64,
                 76.22,  78.83,  81.60,  84.29,  87.10,  89.99,  92.64,  95.15,  97.98, 100.58, 102.89, 105.82, 108.43,
                 110.73, 113.21, 115.77, 117.86, 120.17, 123.21, 125.88, 129.90, 135.98};
      m_ncoll_sys = {0.052, 0.036, 0.104, 0.097, 0.167, 0.228, 0.283, 0.258, 0.296, 0.415,
                     0.492, 0.721, 0.973, 1.211, 1.42,  1.774, 2.047, 2.553, 2.873, 3.198};
      m_b = {10.99, 10.96, 10.93, 10.88, 10.87, 10.85, 10.82, 10.77, 10.74, 10.72, 10.65, 10.58, 10.53, 10.47, 10.39,
             10.36, 10.29, 10.21, 10.16, 10.10, 10.01, 9.95,  9.87,  9.79,  9.71,  9.65,  9.58,  9.51,  9.44,  9.36,
             9.29,  9.21,  9.14,  9.06,  8.98,  8.90,  8.85,  8.76,  8.70,  8.62,  8.54,  8.47,  8.39,  8.31,  8.23,
             8.15,  8.07,  7.99,  7.92,  7.84,  7.75,  7.67,  7.59,  7.51,  7.42,  7.33,  7.26,  7.16,  7.08,  6.99,
             6.90,  6.81,  6.71,  6.63,  6.54,  6.44,  6.34,  6.24,  6.13,  6.04,  5.93,  5.83,  5.73,  5.61,  5.50,
             5.37,  5.27,  5.14,  5.02,  4.89,  4.76,  4.61,  4.48,  4.33,  4.19,  4.04,  3.91,  3.74,  3.62,  3.46,
             3.32,  3.19,  3.04,  2.89,  2.74,  2.62,  2.44,  2.28,  2.06,  1.78};
      m_b_sys = {0.431, 0.172, 0.501, 0.282, 0.351, 0.321, 0.261, 0.162, 0.084, 0.093,
                 0.053, 0.062, 0.082, 0.082, 0.044, 0.035, 0.047, 0.032, 0.026, 0.026};
      if ( m_nclasses == 100 ) {
        m_npart_stat = {0.003, 0.005, 0.005, 0.006, 0.007, 0.008, 0.008, 0.009, 0.010, 0.011, 0.011, 0.012, 0.013,
                        0.014, 0.014, 0.015, 0.016, 0.017, 0.017, 0.018, 0.019, 0.020, 0.020, 0.021, 0.021, 0.022,
                        0.022, 0.023, 0.023, 0.025, 0.025, 0.026, 0.026, 0.027, 0.028, 0.028, 0.029, 0.030, 0.030,
                        0.031, 0.032, 0.033, 0.033, 0.034, 0.035, 0.035, 0.036, 0.037, 0.038, 0.038, 0.040, 0.040,
                        0.041, 0.042, 0.043, 0.044, 0.044, 0.045, 0.046, 0.047, 0.048, 0.049, 0.049, 0.050, 0.052,
                        0.052, 0.053, 0.054, 0.055, 0.055, 0.057, 0.058, 0.059, 0.060, 0.061, 0.063, 0.063, 0.064,
                        0.065, 0.065, 0.066, 0.068, 0.067, 0.069, 0.069, 0.068, 0.070, 0.070, 0.071, 0.070, 0.070,
                        0.070, 0.069, 0.069, 0.069, 0.069, 0.069, 0.069, 0.069, 0.074};
        m_ncoll_stat = {0.003, 0.004, 0.005, 0.006, 0.007, 0.007, 0.008, 0.008, 0.009, 0.009, 0.010, 0.011, 0.011,
                        0.012, 0.013, 0.014, 0.014, 0.015, 0.016, 0.017, 0.018, 0.019, 0.020, 0.020, 0.021, 0.022,
                        0.023, 0.024, 0.025, 0.026, 0.027, 0.028, 0.030, 0.032, 0.033, 0.034, 0.035, 0.037, 0.039,
                        0.041, 0.042, 0.045, 0.046, 0.048, 0.050, 0.053, 0.055, 0.057, 0.060, 0.061, 0.065, 0.066,
                        0.069, 0.073, 0.075, 0.077, 0.080, 0.084, 0.087, 0.091, 0.093, 0.096, 0.100, 0.102, 0.106,
                        0.109, 0.113, 0.119, 0.122, 0.123, 0.128, 0.128, 0.136, 0.138, 0.141, 0.148, 0.149, 0.151,
                        0.157, 0.162, 0.161, 0.166, 0.170, 0.172, 0.174, 0.173, 0.177, 0.175, 0.177, 0.177, 0.175,
                        0.175, 0.172, 0.172, 0.168, 0.165, 0.166, 0.158, 0.159, 0.155};
        m_b_stat     = {0.011, 0.011, 0.011, 0.011, 0.011, 0.011, 0.011, 0.011, 0.011, 0.011, 0.011, 0.010, 0.010,
                    0.010, 0.010, 0.010, 0.010, 0.010, 0.010, 0.010, 0.009, 0.009, 0.009, 0.009, 0.009, 0.009,
                    0.008, 0.008, 0.008, 0.008, 0.008, 0.008, 0.008, 0.007, 0.007, 0.007, 0.007, 0.007, 0.007,
                    0.007, 0.007, 0.007, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006,
                    0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006,
                    0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.006, 0.007, 0.007, 0.007, 0.007, 0.007,
                    0.007, 0.008, 0.008, 0.008, 0.009, 0.009, 0.009, 0.009, 0.010, 0.010, 0.010, 0.010, 0.010,
                    0.010, 0.010, 0.010, 0.010, 0.010, 0.010, 0.009, 0.009, 0.008};
      } else if ( m_nclasses == 50 ) {
        m_npart_stat = {0.003, 0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.010, 0.012, 0.013, 0.014, 0.014, 0.015,
                        0.016, 0.017, 0.018, 0.019, 0.020, 0.021, 0.022, 0.023, 0.024, 0.025, 0.026, 0.027, 0.028,
                        0.030, 0.031, 0.032, 0.033, 0.034, 0.036, 0.037, 0.038, 0.039, 0.041, 0.042, 0.044, 0.045,
                        0.046, 0.048, 0.048, 0.049, 0.050, 0.050, 0.050, 0.049, 0.049, 0.050, 0.054};
        m_ncoll_stat = {0.003, 0.004, 0.005, 0.006, 0.006, 0.007, 0.008, 0.009, 0.011, 0.012, 0.013, 0.014, 0.015,
                        0.017, 0.018, 0.020, 0.022, 0.024, 0.026, 0.028, 0.031, 0.033, 0.037, 0.040, 0.043, 0.046,
                        0.050, 0.054, 0.058, 0.063, 0.067, 0.072, 0.076, 0.082, 0.087, 0.091, 0.097, 0.103, 0.106,
                        0.113, 0.116, 0.121, 0.123, 0.125, 0.126, 0.124, 0.122, 0.118, 0.115, 0.113};
        m_b_stat     = {0.008, 0.008, 0.008, 0.008, 0.008, 0.007, 0.007, 0.007, 0.007, 0.007, 0.007, 0.006, 0.006,
                    0.006, 0.006, 0.006, 0.005, 0.005, 0.005, 0.005, 0.005, 0.005, 0.004, 0.004, 0.004, 0.004,
                    0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.004, 0.005, 0.005, 0.005,
                    0.005, 0.006, 0.006, 0.007, 0.007, 0.007, 0.007, 0.007, 0.007, 0.007, 0.006};
      } else if ( m_nclasses == 25 ) {
        m_npart_stat = {0.003, 0.004, 0.006, 0.007, 0.009, 0.010, 0.011, 0.013, 0.014, 0.015, 0.017, 0.018, 0.020,
                        0.022, 0.024, 0.025, 0.027, 0.029, 0.032, 0.033, 0.035, 0.036, 0.036, 0.036, 0.041};
        m_ncoll_stat = {0.002, 0.004, 0.005, 0.006, 0.008, 0.010, 0.011, 0.014, 0.016, 0.019, 0.023, 0.027, 0.032,
                        0.038, 0.044, 0.050, 0.057, 0.064, 0.072, 0.079, 0.085, 0.089, 0.089, 0.086, 0.083};
        m_b_stat     = {0.005, 0.005, 0.005, 0.005, 0.005, 0.005, 0.004, 0.004, 0.004, 0.003, 0.003, 0.003, 0.003,
                    0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.004, 0.004, 0.005, 0.005, 0.005, 0.005};
      } else if ( m_nclasses == 20 ) {
        m_npart_stat = {0.003, 0.004, 0.006, 0.008, 0.009, 0.011, 0.012, 0.014, 0.016, 0.017,
                        0.019, 0.021, 0.023, 0.026, 0.028, 0.030, 0.032, 0.033, 0.033, 0.038};
        m_ncoll_stat = {0.002, 0.004, 0.005, 0.007, 0.009, 0.011, 0.014, 0.017, 0.021, 0.026,
                        0.032, 0.039, 0.046, 0.054, 0.062, 0.070, 0.078, 0.080, 0.079, 0.076};
        m_b_stat     = {0.005, 0.005, 0.005, 0.004, 0.004, 0.004, 0.003, 0.003, 0.003, 0.003,
                    0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.004, 0.005, 0.005, 0.004};
      } else if ( m_nclasses == 10 ) {
        m_npart_stat = {0.002, 0.005, 0.008, 0.010, 0.013, 0.016, 0.020, 0.024, 0.026, 0.030};
        m_ncoll_stat = {0.002, 0.005, 0.008, 0.012, 0.019, 0.027, 0.039, 0.051, 0.060, 0.059};
        m_b_stat     = {0.003, 0.003, 0.003, 0.002, 0.002, 0.002, 0.002, 0.002, 0.003, 0.003};
      } else if ( m_nclasses == 5 ) {
        m_npart_stat = {0.003, 0.008, 0.014, 0.022, 0.028};
        m_ncoll_stat = {0.003, 0.009, 0.021, 0.040, 0.051};
        m_b_stat     = {0.002, 0.002, 0.002, 0.002, 0.003};
      } else if ( m_nclasses == 4 ) {
        m_npart_stat = {0.004, 0.011, 0.020, 0.028};
        m_ncoll_stat = {0.003, 0.013, 0.033, 0.051};
        m_b_stat     = {0.002, 0.002, 0.002, 0.003};
      } else if ( m_nclasses == 2 ) {
        m_npart_stat = {0.009, 0.031};
        m_ncoll_stat = {0.009, 0.051};
        m_b_stat     = {0.002, 0.003};
      } else if ( m_nclasses == 1 ) {
        m_npart_stat = {0.028};
        m_ncoll_stat = {0.041};
        m_b_stat     = {0.003};
      }
    } else {
      fatal() << Form( "Collisions system %s not found.", m_CollSyst.value().c_str() ) << endmsg;
      return StatusCode::FAILURE;
    }

    for ( Int_t i = 0; i < m_nclasses; i++ ) {
      m_npart_temp.push_back( 0 );
      m_ncoll_temp.push_back( 0 );
      m_b_temp.push_back( 0 );
    }

    m_count = 0;
    for ( Int_t i = 0; i < int( m_npart.size() ); i++ ) {
      if ( i % int( 100.0 / m_nclasses ) == 0 && i != 0 ) { m_count += 1; }
      m_npart_temp[m_count] += m_npart[i];
      m_ncoll_temp[m_count] += m_ncoll[i];
      m_b_temp[m_count] += m_b[i];
    }

    return StatusCode::SUCCESS;
  }
  fatal() << "Number of centrality classes selected not allowed... Must be a divisor of 100." << endmsg;
  return StatusCode::FAILURE;
}
StatusCode TupleToolCentrality::finalize() { return StatusCode::SUCCESS; }
StatusCode TupleToolCentrality::fill( Tuples::Tuple& tuple ) {
  const std::string prefix = fullName();
  LHCb::CaloDigits* digits = getIfExists<LHCb::CaloDigits>( m_DigitLocation );
  if ( !digits ) return StatusCode::SUCCESS;
  for ( const auto& digit : *digits ) {
    const LHCb::CaloCellID cellID = digit->cellID();
    m_index.push_back( cellID.index() );
    m_calo.push_back( cellID.calo() );
    m_area.push_back( cellID.area() );
    m_row.push_back( cellID.row() );
    m_column.push_back( cellID.col() );

    m_es.push_back( digit->e() );
    if ( msgLevel( MSG::DEBUG ) ) debug() << cellID.toString() << " has an energy of " << m_es.back() << " \n";
  }
  if ( msgLevel( MSG::DEBUG ) )
    debug() << " saved " << m_index.size() << " digits to n tuple " + m_CaloName + "Digit." << endmsg;
  if ( m_verbose ) {
    tuple
        ->column( m_extraName + m_CaloName + "DigitEnergySum",
                  std::accumulate( std::begin( m_es ), std::end( m_es ), 0.0 ) )
        .ignore();
  }

  m_TES = std::accumulate( std::begin( m_es ), std::end( m_es ), 0.0 ); // Total Energy Sum

  while ( m_TES > m_ecalcuts[m_i] && m_switch ) { // Find the exact percentile in 1% intervals
    m_i += 1;
    if ( m_i > 99 ) {   // most central events will be higher than last cut value
      m_j = m_nclasses; // assign m_j value so that event falls in the most central interval, ie, m_centrality = 0
                        // (lower bound for the class)
      m_switch = 0;
    }
  }

  if ( m_i == 0 ) { // If deposited energy is 0.0, assign the most peripheral class.
    m_i = 1;
  }

  while ( m_j < ( m_i * 1.0 ) / ( 100.0 / m_nclasses ) ) { // Assign the corresponding centrality class in the given
                                                           // number of classes desired
    m_j += 1;
  }

  m_centrality = ( m_nclasses - m_j + 1 ) *
                 ( 100.0 / m_nclasses ); // centrality class upper bound. To get the lower bound, delete the "+1".
  m_npart_final =
      m_npart_temp[m_j - 1] / ( 100.0 / m_nclasses ); // m_j - 1 because m_j takes values from 1 to m_nclasses
  m_ncoll_final = m_ncoll_temp[m_j - 1] / ( 100.0 / m_nclasses );
  m_b_final     = m_b_temp[m_j - 1] / ( 100.0 / m_nclasses );
  m_npart_error = sqrt( pow( m_npart_sys[( m_i - 1 ) / 5], 2 ) +
                        pow( m_npart_stat[m_j - 1], 2 ) ); // Add in quadrature std of distribution and systematic
                                                           // uncertainty. m_i-1 because the event is located in an
                                                           // energy E < m_ecalcuts[m_i] and E > m_ecalcuts[m_i-1]
  m_ncoll_error = sqrt( pow( m_ncoll_sys[( m_i - 1 ) / 5], 2 ) + pow( m_ncoll_stat[m_j - 1], 2 ) );
  m_b_error     = sqrt( pow( m_b_sys[( m_i - 1 ) / 5], 2 ) + pow( m_b_stat[m_j - 1], 2 ) );

  tuple->column( "CentralityPercentile", m_centrality ).ignore();
  tuple->column( "N_part", m_npart_final ).ignore();
  tuple->column( "N_part_error", m_npart_error ).ignore();
  tuple->column( "N_coll", m_ncoll_final ).ignore();
  tuple->column( "N_coll_error", m_ncoll_error ).ignore();
  tuple->column( "B", m_b_final ).ignore();
  tuple->column( "B_error", m_b_error ).ignore();

  if ( m_debug ) {
    tuple->column( "m_i", m_i ).ignore();
    tuple->column( "m_j", m_j ).ignore();
  }

  m_index.clear();
  m_calo.clear();
  m_area.clear();
  m_row.clear();
  m_column.clear();
  m_es.clear();

  m_i      = 0; // counter for specific centrality percentile in 1% intervals. The higher m_i the more central the event
  m_j      = 0; // counter for specific centrality class asked by the user. The higher m_j the more central the event;
  m_switch = 1;

  return StatusCode::SUCCESS;
}
